
import React from 'react';
import { NavLink, useLocation } from 'react-router-dom';
import { NAV_ITEMS, APP_TITLE, NavItem } from '../../constants.tsx'; // Ensure .tsx

const SidebarNavItem: React.FC<{ item: NavItem; level?: number }> = ({ item, level = 0 }) => {
  const location = useLocation();
  const isChildActive = item.children?.some(child => location.pathname.startsWith(child.path)) ?? false;
  const [isOpen, setIsOpen] = React.useState(isChildActive);

  React.useEffect(() => {
    if (isChildActive) {
      setIsOpen(true);
    }
  }, [isChildActive, location.pathname]);


  const hasChildren = item.children && item.children.length > 0;

  const handleToggleOrNavigate = (e: React.MouseEvent) => {
    if (hasChildren) {
      e.preventDefault(); // Prevent navigation if it's a parent with children
      setIsOpen(!isOpen);
    }
    // If no children, NavLink will handle navigation
  };
  
  const linkPadding = `pl-${4 + level * 2}`; // Indentation for nested items: pl-4, pl-6, pl-8...

  const baseLinkClasses = `flex items-center p-3 ${linkPadding} rounded-lg hover:bg-sky-700 hover:text-white transition-colors duration-200 group`;
  const activeLinkClasses = "bg-sky-600 text-white font-semibold";
  const inactiveLinkClasses = "text-slate-100 hover:text-white";


  const linkContent = (
    <>
      <item.icon className="w-5 h-5 mr-3 shrink-0" />
      <span className="flex-1 text-sm">{item.name}</span>
      {hasChildren && (
        <svg 
            className={`w-4 h-4 transform transition-transform duration-200 ${isOpen ? 'rotate-90' : ''} group-hover:text-white`} 
            fill="currentColor" 
            viewBox="0 0 20 20"
        >
          <path fillRule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clipRule="evenodd" />
        </svg>
      )}
    </>
  );

  return (
    <li>
      {hasChildren ? (
         <a // Using <a> for parents, NavLink for actual navigation links
          href={item.path} // Still include path for context, though click is handled
          onClick={handleToggleOrNavigate}
          className={`${baseLinkClasses} ${isChildActive ? "text-white" : inactiveLinkClasses} w-full text-left`}
          aria-expanded={isOpen}
        >
          {linkContent}
        </a>
      ) : (
        <NavLink
          to={item.path}
          onClick={handleToggleOrNavigate} // NavLink will navigate, toggle is no-op here
          className={({ isActive }) =>
            `${baseLinkClasses} ${isActive ? activeLinkClasses : inactiveLinkClasses}`
          }
        >
          {linkContent}
        </NavLink>
      )}
      {hasChildren && isOpen && (
        <ul className="mt-1 space-y-1">
          {item.children?.map((child) => (
            <SidebarNavItem key={child.name} item={child} level={level + 1}/>
          ))}
        </ul>
      )}
    </li>
  );
};


const Sidebar: React.FC = () => {
  return (
    <aside className="w-60 sm:w-64 bg-sky-800 text-white flex-col p-4 space-y-4 overflow-y-auto hidden md:flex print:hidden">
      <div className="text-left py-3 border-b border-sky-700">
        <h2 className="text-xl font-bold text-white">{APP_TITLE}</h2>
        <p className="text-xs text-sky-200">Administrare Locală</p>
      </div>
      <nav className="flex-grow">
        <ul className="space-y-1.5">
          {NAV_ITEMS.map((item) => (
            <SidebarNavItem key={item.name} item={item} />
          ))}
        </ul>
      </nav>
      <div className="mt-auto text-center text-xs text-sky-300 p-2 border-t border-sky-700">
        © {new Date().getFullYear()} Registru Local
      </div>
    </aside>
  );
};

export default Sidebar;
