
import React from 'react';

type AlertType = 'success' | 'error' | 'warning' | 'info';

interface AlertProps {
  type: AlertType;
  message: string | React.ReactNode;
  title?: string;
  onClose?: () => void;
  className?: string;
}

const Alert: React.FC<AlertProps> = ({ type, title, message, onClose, className = '' }) => {
  const baseClasses = "p-4 mb-4 rounded-md flex items-start shadow-md";
  const typeStyles = {
    success: {
      container: "bg-green-50 border border-green-300 text-green-700",
      iconBg: "bg-green-100",
      iconColor: "text-green-600",
      titleColor: "text-green-800",
      closeHover: "hover:bg-green-100",
    },
    error: {
      container: "bg-red-50 border border-red-300 text-red-700",
      iconBg: "bg-red-100",
      iconColor: "text-red-600",
      titleColor: "text-red-800",
      closeHover: "hover:bg-red-100",
    },
    warning: {
      container: "bg-yellow-50 border border-yellow-300 text-yellow-700",
      iconBg: "bg-yellow-100",
      iconColor: "text-yellow-600",
      titleColor: "text-yellow-800",
      closeHover: "hover:bg-yellow-100",
    },
    info: {
      container: "bg-sky-50 border border-sky-300 text-sky-700",
      iconBg: "bg-sky-100",
      iconColor: "text-sky-600",
      titleColor: "text-sky-800",
      closeHover: "hover:bg-sky-100",
    },
  };

  const currentStyle = typeStyles[type];

  const Icon: React.FC<{ type: AlertType }> = ({ type }) => {
    const iconBaseClass = `w-5 h-5 shrink-0 ${currentStyle.iconColor}`;
    if (type === 'success') return <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className={iconBaseClass}><path strokeLinecap="round" strokeLinejoin="round" d="M9 12.75L11.25 15 15 9.75M21 12a9 9 0 11-18 0 9 9 0 0118 0z" /></svg>;
    if (type === 'error') return <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className={iconBaseClass}><path strokeLinecap="round" strokeLinejoin="round" d="M12 9v3.75m9-.75a9 9 0 11-18 0 9 9 0 0118 0zm-9 3.75h.008v.008H12v-.008z" /></svg>; // Alternative error icon: M10.293 3.293a1 1 0 00-1.414 0L3 9.586V10a1 1 0 001 1h12a1 1 0 001-1v-.414l-5.879-6.293a1 1 0 00-1.414 0L12 5.586l-1.707-1.293zM3 14a1 1 0 011-1h12a1 1 0 011 1v2a1 1 0 01-1 1H4a1 1 0 01-1-1v-2z
    if (type === 'warning') return <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className={iconBaseClass}><path strokeLinecap="round" strokeLinejoin="round" d="M12 9v3.75m-9.303 3.376c-.866 1.5.217 3.374 1.948 3.374h14.71c1.73 0 2.813-1.874 1.948-3.374L13.949 3.378c-.866-1.5-3.032-1.5-3.898 0L2.697 16.126zM12 15.75h.007v.008H12v-.008z" /></svg>;
    if (type === 'info') return <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={1.5} stroke="currentColor" className={iconBaseClass}><path strokeLinecap="round" strokeLinejoin="round" d="M11.25 11.25l.041-.02a.75.75 0 011.063.852l-.708 2.836a.75.75 0 001.063.853l.041-.021M21 12a9 9 0 11-18 0 9 9 0 0118 0zm-9-3.75h.008v.008H12V8.25z" /></svg>;
    return null;
  };

  return (
    <div className={`${baseClasses} ${currentStyle.container} ${className}`} role="alert">
      <div className={`p-1.5 rounded-full ${currentStyle.iconBg} mr-3`}>
        <Icon type={type} />
      </div>
      <div className="flex-1">
        {title && <h4 className={`text-sm font-semibold ${currentStyle.titleColor} mb-1`}>{title}</h4>}
        <div className="text-sm">{message}</div>
      </div>
      {onClose && (
        <button 
            onClick={onClose} 
            className={`ml-auto -mx-1.5 -my-1.5 p-1.5 rounded-md ${currentStyle.closeHover} ${currentStyle.iconColor} focus:outline-none focus:ring-2 focus:ring-offset-1 focus:ring-current`}
            aria-label="Dismiss alert"
        >
           <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={2} stroke="currentColor" className="w-5 h-5">
            <path strokeLinecap="round" strokeLinejoin="round" d="M6 18L18 6M6 6l12 12" />
          </svg>
        </button>
      )}
    </div>
  );
};

export default Alert;
