
import React from 'react';

type ButtonVariant = 'primary' | 'secondary' | 'danger' | 'success' | 'outline' | 'ghost';
type ButtonSize = 'sm' | 'md' | 'lg' | 'icon';

interface ButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  variant?: ButtonVariant;
  size?: ButtonSize;
  isLoading?: boolean;
  leftIcon?: React.ReactNode;
  rightIcon?: React.ReactNode;
  children?: React.ReactNode; // Made optional for icon-only buttons
}

const Button: React.FC<ButtonProps> = ({
  children,
  variant = 'primary',
  size = 'md',
  isLoading = false,
  leftIcon,
  rightIcon,
  className = '',
  ...props
}) => {
  const baseStyles = "font-semibold rounded-md focus:outline-none focus-visible:ring-2 focus-visible:ring-offset-2 transition-all duration-150 ease-in-out inline-flex items-center justify-center shadow-sm disabled:shadow-none";

  const sizeStyles = {
    sm: "px-3 py-1.5 text-xs",
    md: "px-4 py-2 text-sm",
    lg: "px-5 py-2.5 text-base",
    icon: "p-2", // For icon-only buttons
  };

  const variantStyles = {
    primary: "bg-sky-600 text-white hover:bg-sky-700 focus-visible:ring-sky-500 disabled:bg-sky-400",
    secondary: "bg-slate-200 text-slate-800 hover:bg-slate-300 focus-visible:ring-slate-500 disabled:bg-slate-100 disabled:text-slate-400 border border-slate-300 hover:border-slate-400",
    danger: "bg-red-600 text-white hover:bg-red-700 focus-visible:ring-red-500 disabled:bg-red-400",
    success: "bg-green-600 text-white hover:bg-green-700 focus-visible:ring-green-500 disabled:bg-green-400",
    outline: "bg-transparent border border-sky-600 text-sky-700 hover:bg-sky-50 focus-visible:ring-sky-500 disabled:text-sky-400 disabled:border-sky-400 disabled:bg-transparent",
    ghost: "bg-transparent text-sky-700 hover:bg-sky-100 focus-visible:ring-sky-500 disabled:text-slate-400 hover:text-sky-800"
  };
  
  const disabledStyles = "opacity-70 cursor-not-allowed";

  return (
    <button
      className={`${baseStyles} ${sizeStyles[size]} ${variantStyles[variant]} ${isLoading || props.disabled ? disabledStyles : ''} ${className}`}
      disabled={isLoading || props.disabled}
      {...props}
    >
      {isLoading && (
        <svg className={`animate-spin h-5 w-5 ${children ? '-ml-1 mr-2' : ''}`} xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
          <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
          <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
        </svg>
      )}
      {leftIcon && !isLoading && <span className={`${children ? 'mr-2' : ''} h-5 w-5 inline-flex items-center justify-center`}>{leftIcon}</span>}
      {children}
      {rightIcon && !isLoading && <span className={`${children ? 'ml-2' : ''} h-5 w-5 inline-flex items-center justify-center`}>{rightIcon}</span>}
    </button>
  );
};

export default Button;
