import React from 'react';
import { BuildingLibraryIcon, UsersIcon, BuildingOfficeIcon, ArchiveBoxIcon } from '../../constants.tsx'; // Ensure .tsx

interface StatCardProps {
  title: string;
  value: string | number;
  icon: React.ReactElement<React.SVGProps<SVGSVGElement>>;
  colorScheme?: string; // e.g., 'sky', 'green', 'amber', 'purple'
  description?: string;
}

const StatCard: React.FC<StatCardProps> = ({ title, value, icon, colorScheme = 'sky', description }) => {
  const colors = {
    sky: { border: 'border-sky-500', bg: 'bg-sky-50', text: 'text-sky-600', iconBg: 'bg-sky-100' },
    green: { border: 'border-green-500', bg: 'bg-green-50', text: 'text-green-600', iconBg: 'bg-green-100' },
    amber: { border: 'border-amber-500', bg: 'bg-amber-50', text: 'text-amber-600', iconBg: 'bg-amber-100' },
    purple: { border: 'border-purple-500', bg: 'bg-purple-50', text: 'text-purple-600', iconBg: 'bg-purple-100' },
    red: { border: 'border-red-500', bg: 'bg-red-50', text: 'text-red-600', iconBg: 'bg-red-100' },
  };
  const selectedColor = colors[colorScheme as keyof typeof colors] || colors.sky;

  return (
    <div className={`bg-white p-5 rounded-xl shadow-lg flex items-start space-x-4 border-l-4 ${selectedColor.border} hover:shadow-xl transition-shadow`}>
      <div className={`p-3 rounded-full ${selectedColor.iconBg} shrink-0`}>
        {React.cloneElement(icon, { className: `w-7 h-7 ${selectedColor.text}` })}
      </div>
      <div>
        <p className="text-sm font-medium text-slate-500">{title}</p>
        <p className="text-2xl font-semibold text-slate-800">{value}</p>
        {description && <p className="text-xs text-slate-400 mt-1">{description}</p>}
      </div>
    </div>
  );
};

const DashboardPage: React.FC = () => {
  // Mock data - in a real app, this would come from API calls
  const stats = {
    totalProtopopiate: 3,
    totalUsers: 4,
    activeParohii: 2,
    inventoryItems: 178, // Example value
  };

  const recentActivities = [
    { id: 1, user: "Admin Principal", action: "a adăugat Protopopiatul 'Nord'", time: "acum 2 ore", type: "add" },
    { id: 2, user: "Vasile Ionescu (Paroh)", action: "a actualizat elementul de inventar #INV0123", time: "acum 5 ore", type: "update" },
    { id: 3, user: "Admin Principal", action: "a dezactivat utilizatorul 'user.inactiv'", time: "ieri", type: "delete" },
    { id: 4, user: "Ion Popescu (Protopop)", action: "a vizualizat raportul lunar", time: "ieri", type: "view" },
  ];

  return (
    <div className="space-y-6 sm:space-y-8">
      <h2 className="text-2xl sm:text-3xl font-bold text-slate-800">Panou Principal</h2>
      
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-5">
        <StatCard 
          title="Total Protopopiate" 
          value={stats.totalProtopopiate}
          icon={<BuildingLibraryIcon />}
          colorScheme="sky"
          description="Numărul total de protopopiate înregistrate."
        />
        <StatCard 
          title="Total Utilizatori" 
          value={stats.totalUsers}
          icon={<UsersIcon />}
          colorScheme="green"
          description="Utilizatori cu acces la sistem."
        />
        <StatCard 
          title="Parohii Active" 
          value={stats.activeParohii}
          icon={<BuildingOfficeIcon />}
          colorScheme="amber"
          description="Parohii cu status 'activ'."
        />
        <StatCard 
          title="Articole Inventar" 
          value={stats.inventoryItems}
          icon={<ArchiveBoxIcon />}
          colorScheme="purple"
          description="Bunuri înregistrate în inventar."
        />
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <div className="lg:col-span-2 bg-white p-5 sm:p-6 rounded-xl shadow-lg">
          <h3 className="text-lg sm:text-xl font-semibold text-slate-700 mb-4">Activități Recente</h3>
          {recentActivities.length > 0 ? (
            <ul className="space-y-3 max-h-96 overflow-y-auto pr-2">
              {recentActivities.map((activity) => (
                <li key={activity.id} className="flex items-start p-3 bg-slate-50 hover:bg-slate-100 rounded-md transition-colors">
                  <div className={`mr-3 mt-1 w-2 h-2 rounded-full shrink-0 ${activity.type === 'add' ? 'bg-green-500' : activity.type === 'update' ? 'bg-sky-500' : activity.type === 'delete' ? 'bg-red-500' : 'bg-slate-400'}`}></div>
                  <div className="flex-grow">
                    <span className="font-medium text-slate-700 text-sm">{activity.user}</span>
                    <span className="text-slate-500 text-sm"> {activity.action}</span>
                  </div>
                  <span className="text-xs text-slate-400 shrink-0 ml-2">{activity.time}</span>
                </li>
              ))}
            </ul>
          ) : (
            <p className="text-sm text-slate-500">Nu sunt activități recente de afișat.</p>
          )}
        </div>

        <div className="bg-white p-5 sm:p-6 rounded-xl shadow-lg">
          <h3 className="text-lg sm:text-xl font-semibold text-slate-700 mb-4">Stare Sistem</h3>
          <div className="space-y-3">
            <p className="text-sm text-slate-600 flex justify-between">API Backend: <span className="font-semibold text-green-600">Operațional (Mock)</span></p>
            <p className="text-sm text-slate-600 flex justify-between">Bază de Date: <span className="font-semibold text-green-600">Conectată (Simulat)</span></p>
            <p className="text-sm text-slate-600 flex justify-between">Ultimul Backup: <span className="font-semibold text-slate-700">Azi, 02:00 AM</span></p>
             <hr className="my-3"/>
            <p className="text-xs text-slate-500">Acesta este un panou demonstrativ. Datele sunt simulate pentru testare locală.</p>
          </div>
           <img src="https://picsum.photos/seed/system/400/200?grayscale" alt="Abstract system representation" className="mt-4 rounded-md shadow-sm w-full h-auto object-cover opacity-80"/>
        </div>
      </div>
    </div>
  );
};

export default DashboardPage;