
import React, { useState, useEffect } from 'react';
import { Protopopiat, ProtopopiateStatus, DropdownOption } from '../../types.tsx'; // Ensure .tsx
import Input from '../../components/shared/Input.tsx';
import Button from '../../components/shared/Button.tsx';
import Select from '../../components/shared/Select.tsx';
import * as apiService from '../../services/apiService.ts'; // Ensure .tsx


interface ProtopopiatFormProps {
  initialData?: Protopopiat | null;
  onSubmit: (data: Omit<Protopopiat, 'id_proterie'> | Protopopiat) => Promise<void>; // Make onSubmit async
  onCancel: () => void;
  isSubmitting?: boolean;
}

const ProtopopiatForm: React.FC<ProtopopiatFormProps> = ({ initialData, onSubmit, onCancel, isSubmitting }) => {
  const [formData, setFormData] = useState<Partial<Protopopiat>>({
    denumire: '',
    cui: '',
    adresa: '',
    localitate: '',
    judet: '',
    anul_infiintarii: null, // Use null for nullable numbers
    cod_caen: '',
    descr_caen: '',
    observatii: '',
    status: ProtopopiateStatus.ACTIV,
  });
  
  const [statusOptions, setStatusOptions] = useState<DropdownOption[]>([]);
  
  useEffect(() => {
    if (initialData) {
      setFormData({ 
        ...initialData,
        anul_infiintarii: initialData.anul_infiintarii === undefined ? null : initialData.anul_infiintarii // Ensure null if undefined
      });
    } else {
      // Reset for new form
      setFormData({
        denumire: '', cui: '', adresa: '', localitate: '', judet: '',
        anul_infiintarii: null, cod_caen: '', descr_caen: '', observatii: '',
        status: ProtopopiateStatus.ACTIV,
      });
    }
  }, [initialData]);
  
  useEffect(() => {
    // Options are now { value: "activ", label: "Activ" } etc.
    setStatusOptions(apiService.getProtopopiateStatusOptions());
  }, []);


  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    
    let processedValue: string | number | ProtopopiateStatus | null = value;

    if (e.target instanceof HTMLInputElement && e.target.type === 'number') {
        processedValue = value === '' ? null : parseInt(value, 10); // Allow empty string to become null
    } else if (name === 'status') {
        processedValue = value as ProtopopiateStatus;
    }

    setFormData(prev => ({ ...prev, [name]: processedValue }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    // Basic validation example
    if (!formData.denumire || !formData.localitate) {
        // This alert is basic; consider a more integrated validation message system
        window.alert("Denumirea și Localitatea sunt câmpuri obligatorii."); 
        return;
    }
    // Ensure anul_infiintarii is a number or null
    const dataToSubmit = {
      ...formData,
      anul_infiintarii: formData.anul_infiintarii ? Number(formData.anul_infiintarii) : null,
    };
    await onSubmit(dataToSubmit as Protopopiat); 
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4 sm:space-y-5">
      <Input name="denumire" label="Denumire Protopopiat" value={formData.denumire || ''} onChange={handleChange} required maxLength={150} />
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <Input name="cui" label="CUI (Cod Unic de Înregistrare)" value={formData.cui || ''} onChange={handleChange} maxLength={13} />
        <Input name="anul_infiintarii" label="Anul Înființării" type="number" value={formData.anul_infiintarii?.toString() || ''} onChange={handleChange} min="1000" max={new Date().getFullYear()}/>
      </div>
      <Input name="adresa" label="Adresă Sediu" value={formData.adresa || ''} onChange={handleChange} maxLength={200} />
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <Input name="localitate" label="Localitate" value={formData.localitate || ''} onChange={handleChange} required maxLength={150}/>
        <Input name="judet" label="Județ" value={formData.judet || ''} onChange={handleChange} maxLength={150}/>
      </div>
       <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <Input name="cod_caen" label="Cod CAEN Principal" value={formData.cod_caen || ''} onChange={handleChange} maxLength={10} />
        <Select
            name="status"
            label="Status Operațional"
            value={formData.status || ''} // Default to empty string if null/undefined to match placeholder
            onChange={handleChange}
            options={statusOptions}
            required
        />
      </div>
      <Input as="textarea" name="descr_caen" label="Descriere Activitate CAEN" value={formData.descr_caen || ''} onChange={handleChange} rows={2} />
      <Input as="textarea" name="observatii" label="Observații Generale" value={formData.observatii || ''} onChange={handleChange} rows={3}/>
      
      {/* Footer is now handled by Modal component's footer prop */}
      <div className="flex justify-end space-x-3 pt-3">
        <Button type="button" variant="secondary" onClick={onCancel} disabled={isSubmitting}>
          Anulează
        </Button>
        <Button type="submit" variant="primary" isLoading={isSubmitting} disabled={isSubmitting}>
          {initialData?.id_proterie ? 'Actualizează' : 'Creează'} Protopopiat
        </Button>
      </div>
    </form>
  );
};

export default ProtopopiatForm;
