import React, { useState, useEffect, useCallback } from 'react';
import { Protopopiat, TableColumn, ProtopopiateStatus } from '../../types.tsx'; // Ensure .tsx
import * as apiService from '../../services/apiService.ts'; // Ensure .tsx
import Table from '../../components/shared/Table.tsx';
import Button from '../../components/shared/Button.tsx';
import Modal from '../../components/shared/Modal.tsx';
import LoadingSpinner from '../../components/shared/LoadingSpinner.tsx';
import Alert from '../../components/shared/Alert.tsx';
import ProtopopiatForm from './ProtopopiatForm.tsx'; // Ensure .tsx
import { PlusIcon, PencilIcon, TrashIcon } from '../../constants.tsx';

const ProtopopiatePage: React.FC = () => {
  const [protopopiate, setProtopopiate] = useState<Protopopiat[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [isSubmitting, setIsSubmitting] = useState(false); // For form submission
  const [error, setError] = useState<string | null>(null);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedProtopopiat, setSelectedProtopopiat] = useState<Protopopiat | null>(null);
  const [alertInfo, setAlertInfo] = useState<{ type: 'success' | 'error'; message: string } | null>(null);

  const fetchProtopopiate = useCallback(async () => {
    setIsLoading(true);
    setError(null);
    try {
      const data = await apiService.getProtopopiate();
      setProtopopiate(data);
    } catch (err) {
      const errorMessage = (err as Error).message || 'A apărut o eroare la preluarea datelor.';
      setError(`Eroare la preluarea protopopiatelor: ${errorMessage}`);
      console.error(err);
    } finally {
      setIsLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchProtopopiate();
  }, [fetchProtopopiate]);

  const handleAdd = () => {
    setSelectedProtopopiat(null);
    setAlertInfo(null); // Clear previous alerts
    setIsModalOpen(true);
  };

  const handleEdit = (item: Protopopiat) => {
    setSelectedProtopopiat(item);
    setAlertInfo(null); // Clear previous alerts
    setIsModalOpen(true);
  };

  const handleDelete = async (id: number) => {
    if (window.confirm('Sunteți sigur că doriți să ștergeți acest protopopiat? Această acțiune este ireversibilă.')) {
      setIsSubmitting(true); // Use submitting state for delete operation as well
      setAlertInfo(null);
      try {
        await apiService.deleteProtopopiat(id);
        setAlertInfo({ type: 'success', message: 'Protopopiat șters cu succes!' });
        fetchProtopopiate(); // Refresh list
      } catch (err) {
        const errorMessage = (err as Error).message || 'A apărut o eroare.';
        setAlertInfo({ type: 'error', message: `Eroare la ștergerea protopopiatului: ${errorMessage}` });
        console.error(err);
      } finally {
        setIsSubmitting(false);
      }
    }
  };

  const handleFormSubmit = async (formData: Omit<Protopopiat, 'id_proterie'> | Protopopiat) => {
    setIsSubmitting(true);
    setAlertInfo(null);
    try {
      if ('id_proterie' in formData && formData.id_proterie) {
        await apiService.updateProtopopiat(formData.id_proterie, formData);
        setAlertInfo({ type: 'success', message: 'Protopopiat actualizat cu succes!' });
      } else {
        await apiService.createProtopopiat(formData as Omit<Protopopiat, 'id_proterie'>);
        setAlertInfo({ type: 'success', message: 'Protopopiat creat cu succes!' });
      }
      fetchProtopopiate();
      setIsModalOpen(false);
      setSelectedProtopopiat(null);
    } catch (err) {
      const errorMessage = (err as Error).message || 'A apărut o eroare.';
      setAlertInfo({ type: 'error', message: `Eroare la salvarea protopopiatului: ${errorMessage}` });
      console.error(err);
      // Optionally, keep the modal open on error if desired, by not calling setIsModalOpen(false)
    } finally {
      setIsSubmitting(false);
    }
  };
  
  const columns: TableColumn<Protopopiat>[] = [
    { key: 'id_proterie', header: 'ID' },
    { key: 'denumire', header: 'Denumire' },
    { key: 'localitate', header: 'Localitate' },
    { key: 'judet', header: 'Județ' },
    { key: 'status', header: 'Status', render: (item) => (
        <span className={`px-2.5 py-0.5 text-xs font-medium rounded-full inline-block ${
            item.status === ProtopopiateStatus.ACTIV 
              ? 'bg-green-100 text-green-700 border border-green-300' 
              : 'bg-red-100 text-red-700 border border-red-300'
        }`}>
            {item.status ? item.status.charAt(0).toUpperCase() + item.status.slice(1) : 'N/A'}
        </span>
    )},
    {
      key: 'actions',
      header: 'Acțiuni',
      render: (item) => (
        <div className="space-x-2 flex">
          <Button variant="outline" size="sm" onClick={() => handleEdit(item)} aria-label={`Editează ${item.denumire}`}>
             <PencilIcon className="w-4 h-4 mr-1" />
            Edit
          </Button>
          <Button variant="danger" size="sm" onClick={() => handleDelete(item.id_proterie)} aria-label={`Șterge ${item.denumire}`}>
             <TrashIcon className="w-4 h-4 mr-1" />
            Șterge
          </Button>
        </div>
      ),
    },
  ];

  if (isLoading && protopopiate.length === 0) {
    return <LoadingSpinner message="Se încarcă protopopiatele..." size="lg" className="h-full" />;
  }

  return (
    <div className="space-y-6">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
        <h2 className="text-2xl sm:text-3xl font-bold text-slate-800">Administrare Protopopiate</h2>
        <Button variant="primary" onClick={handleAdd}>
           <PlusIcon className="w-5 h-5 mr-2" />
          Adaugă Protopopiat
        </Button>
      </div>

      {alertInfo && <Alert type={alertInfo.type} message={alertInfo.message} onClose={() => setAlertInfo(null)} />}
      {error && !alertInfo && <Alert type="error" message={error} onClose={() => setError(null)} />} 

      {isSubmitting && <Alert type="info" message="Se procesează..." />}
      
      <Table columns={columns} data={protopopiate} getKey={(item) => item.id_proterie} caption="Lista Protopopiatelor Înregistrate"/>

      <Modal
        isOpen={isModalOpen}
        onClose={() => { setIsModalOpen(false); setSelectedProtopopiat(null); }}
        title={selectedProtopopiat ? 'Editează Protopopiat' : 'Adaugă Protopopiat Nou'}
        size="lg"
      >
        <ProtopopiatForm
          initialData={selectedProtopopiat}
          onSubmit={handleFormSubmit}
          onCancel={() => { setIsModalOpen(false); setSelectedProtopopiat(null); }}
          isSubmitting={isSubmitting}
        />
      </Modal>
    </div>
  );
};

export default ProtopopiatePage;