import React, { useState, useEffect } from 'react';
import { User, UserStatus, Rol, Parohie, Protopopiat, DropdownOption } from '../../types.tsx'; // Ensure .tsx
import Input from '../../components/shared/Input.tsx';
import Button from '../../components/shared/Button.tsx';
import Select from '../../components/shared/Select.tsx';
import * as apiService from '../../services/apiService.ts'; // Ensure .tsx

interface UserFormProps {
  initialData?: User | null;
  onSubmit: (data: Omit<User, 'id_user' | 'last_login'> | User) => Promise<void>;
  onCancel: () => void;
  roluri: Rol[];
  parohii: Parohie[];
  protopopiate: Protopopiat[];
  isSubmitting?: boolean;
}

const UserForm: React.FC<UserFormProps> = ({ 
    initialData, 
    onSubmit, 
    onCancel, 
    roluri, 
    parohii, 
    protopopiate,
    isSubmitting 
}) => {
  const [formData, setFormData] = useState<Partial<User>>({
    username: '',
    password: '',
    Nume: '',
    id_rol: undefined, // Use undefined for "not selected"
    status: UserStatus.ACTIV,
    id_parohie: undefined,
    id_proterie: undefined,
  });
  
  const [statusOptions, setStatusOptions] = useState<DropdownOption[]>([]);
  
  useEffect(() => {
     if (initialData) {
      setFormData({ 
        ...initialData, 
        password: '', // Clear password for edit form; only set if changed
        id_rol: initialData.id_rol ?? undefined,
        id_parohie: initialData.id_parohie ?? undefined,
        id_proterie: initialData.id_proterie ?? undefined,
      });
    } else {
      // Reset for new form
      setFormData({
        username: '', password: '', Nume: '', id_rol: undefined,
        status: UserStatus.ACTIV, id_parohie: undefined, id_proterie: undefined,
      });
    }
  }, [initialData]);
  
  useEffect(() => {
    setStatusOptions(apiService.getUserStatusOptions());
  }, []);

  const rolOptions: DropdownOption[] = roluri.map(r => ({ value: r.id_rol, label: `${r.nume_rol} (${r.descriere || 'Rol'})` }));
  const parohieOptions: DropdownOption[] = parohii.map(p => ({ value: p.id_parohie, label: p.nume_parohie || 'Parohie N/A' }));
  const protopopiatOptions: DropdownOption[] = protopopiate.map(pr => ({ value: pr.id_proterie, label: pr.denumire || 'Protopopiat N/A' }));

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
     let processedValue: string | number | UserStatus | null | undefined = value;

    if (name === 'id_rol' || name === 'id_parohie' || name === 'id_proterie') {
        processedValue = value ? parseInt(value, 10) : undefined; 
    } else if (name === 'status') {
        processedValue = value as UserStatus;
    }

    setFormData(prev => ({ ...prev, [name]: processedValue }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!formData.username || !formData.Nume || !formData.id_rol) {
      window.alert("Nume utilizator, Numele complet și Rolul sunt câmpuri obligatorii.");
      return;
    }
    if (!initialData?.id_user && !formData.password) { // Password required for new users
        window.alert("Parola este obligatorie pentru utilizatorii noi.");
        return;
    }
    
    const payload = {
        ...formData,
        id_rol: formData.id_rol ? Number(formData.id_rol) : null,
        id_parohie: formData.id_parohie ? Number(formData.id_parohie) : null,
        id_proterie: formData.id_proterie ? Number(formData.id_proterie) : null,
    };
    
    // If password is an empty string and it's an update, don't send it
    if (initialData?.id_user && payload.password === '') {
        delete payload.password;
    }

    await onSubmit(payload as User);
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4 sm:space-y-5">
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <Input name="username" label="Nume Utilizator (Login)" value={formData.username || ''} onChange={handleChange} required maxLength={150}/>
        <Input name="Nume" label="Nume Complet Afișat" value={formData.Nume || ''} onChange={handleChange} required maxLength={150}/>
      </div>
      
      <Input 
        name="password" 
        label={initialData?.id_user ? "Parolă Nouă (lăsați gol pentru a păstra actuala)" : "Parolă"}
        type="password" 
        value={formData.password || ''} 
        onChange={handleChange} 
        required={!initialData?.id_user} 
        autoComplete="new-password"
      />
      
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <Select name="id_rol" label="Rol Utilizator" value={formData.id_rol?.toString() || ''} onChange={handleChange} options={rolOptions} required />
        <Select name="status" label="Status Utilizator" value={formData.status || ''} onChange={handleChange} options={statusOptions} required />
      </div>
      
      <h4 className="text-sm font-medium text-slate-600 pt-2 border-t border-slate-200">Asignări Opționale</h4>
      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <Select name="id_parohie" label="Parohie Asignată" value={formData.id_parohie?.toString() || ''} onChange={handleChange} options={parohieOptions} placeholderOption="Nicio parohie selectată"/>
        <Select name="id_proterie" label="Protopopiat Asignat" value={formData.id_proterie?.toString() || ''} onChange={handleChange} options={protopopiatOptions} placeholderOption="Niciun protopopiat selectat"/>
      </div>
      
      <div className="flex justify-end space-x-3 pt-3">
        <Button type="button" variant="secondary" onClick={onCancel} disabled={isSubmitting}>
          Anulează
        </Button>
        <Button type="submit" variant="primary" isLoading={isSubmitting} disabled={isSubmitting}>
          {initialData?.id_user ? 'Actualizează' : 'Creează'} Utilizator
        </Button>
      </div>
    </form>
  );
};

export default UserForm;