import React, { useState, useEffect, useCallback } from 'react';
import { User, TableColumn, Rol, Parohie, Protopopiat, UserStatus } from '../../types.tsx'; // Ensure .tsx
import * as apiService from '../../services/apiService.ts'; // Ensure .tsx
import Table from '../../components/shared/Table.tsx';
import Button from '../../components/shared/Button.tsx';
import Modal from '../../components/shared/Modal.tsx';
import LoadingSpinner from '../../components/shared/LoadingSpinner.tsx';
import Alert from '../../components/shared/Alert.tsx';
import UserForm from './UserForm.tsx'; // Ensure .tsx
import { PlusIcon, PencilIcon, TrashIcon } from '../../constants.tsx';

const UsersPage: React.FC = () => {
  const [users, setUsers] = useState<User[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedUser, setSelectedUser] = useState<User | null>(null);
  const [alertInfo, setAlertInfo] = useState<{ type: 'success' | 'error'; message: string } | null>(null);

  // Data for dropdowns
  const [roluri, setRoluri] = useState<Rol[]>([]);
  const [parohii, setParohii] = useState<Parohie[]>([]);
  const [protopopiate, setProtopopiate] = useState<Protopopiat[]>([]);

  const fetchData = useCallback(async () => {
    setIsLoading(true);
    setError(null);
    try {
      const [usersData, roluriData, parohiiData, protopopiateData] = await Promise.all([
        apiService.getUsers(),
        apiService.getRoluri(),
        apiService.getParohiiList(),
        apiService.getProtopopiateList()
      ]);
      setUsers(usersData);
      setRoluri(roluriData);
      setParohii(parohiiData);
      setProtopopiate(protopopiateData);
    } catch (err) {
      const errorMessage = (err as Error).message || 'A apărut o eroare la preluarea datelor.';
      setError(`Eroare la preluarea datelor utilizatorilor: ${errorMessage}`);
      console.error(err);
    } finally {
      setIsLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  const handleAdd = () => {
    setSelectedUser(null);
    setAlertInfo(null);
    setIsModalOpen(true);
  };

  const handleEdit = (item: User) => {
    setSelectedUser(item);
    setAlertInfo(null);
    setIsModalOpen(true);
  };

  const handleDelete = async (id: number) => {
    if (window.confirm('Sunteți sigur că doriți să ștergeți acest utilizator? Acțiunea este ireversibilă.')) {
      setIsSubmitting(true);
      setAlertInfo(null);
      try {
        await apiService.deleteUser(id);
        setAlertInfo({ type: 'success', message: 'Utilizator șters cu succes!' });
        fetchData(); 
      } catch (err) {
        const errorMessage = (err as Error).message || 'A apărut o eroare.';
        setAlertInfo({ type: 'error', message: `Eroare la ștergerea utilizatorului: ${errorMessage}` });
        console.error(err);
      } finally {
        setIsSubmitting(false);
      }
    }
  };

  const handleFormSubmit = async (formData: Omit<User, 'id_user' | 'last_login'> | User) => {
    setIsSubmitting(true);
    setAlertInfo(null);
    try {
      const dataToSubmit = {...formData};
      if ('id_user' in dataToSubmit && dataToSubmit.id_user) {
        // For update, remove password if it's empty (means not changed)
        if (dataToSubmit.password === '') {
          delete dataToSubmit.password;
        }
        await apiService.updateUser(dataToSubmit.id_user, dataToSubmit);
        setAlertInfo({ type: 'success', message: 'Utilizator actualizat cu succes!' });
      } else {
        await apiService.createUser(dataToSubmit as Omit<User, 'id_user' | 'last_login'>);
        setAlertInfo({ type: 'success', message: 'Utilizator creat cu succes!' });
      }
      fetchData();
      setIsModalOpen(false);
      setSelectedUser(null);
    } catch (err) {
      const errorMessage = (err as Error).message || 'A apărut o eroare.';
      setAlertInfo({ type: 'error', message: `Eroare la salvarea utilizatorului: ${errorMessage}` });
      console.error(err);
    } finally {
      setIsSubmitting(false);
    }
  };

  const getRolName = (id_rol: number | null) => roluri.find(r => r.id_rol === id_rol)?.nume_rol || 'N/A';
  const getParohieName = (id_parohie: number | null) => parohii.find(p => p.id_parohie === id_parohie)?.nume_parohie || '-';
  const getProtopopiatName = (id_proterie: number | null) => protopopiate.find(p => p.id_proterie === id_proterie)?.denumire || '-';
  
  const columns: TableColumn<User>[] = [
    { key: 'id_user', header: 'ID' },
    { key: 'username', header: 'Nume Utilizator' },
    { key: 'Nume', header: 'Nume Complet' },
    { key: 'id_rol', header: 'Rol', render: (item) => getRolName(item.id_rol) },
    { key: 'status', header: 'Status', render: (item) => (
        <span className={`px-2.5 py-0.5 text-xs font-medium rounded-full inline-block ${
            item.status === UserStatus.ACTIV 
              ? 'bg-green-100 text-green-700 border border-green-300' 
              : 'bg-red-100 text-red-700 border border-red-300'
        }`}>
            {item.status ? item.status.charAt(0).toUpperCase() + item.status.slice(1) : 'N/A'}
        </span>
    )},
    { key: 'id_parohie', header: 'Parohie Asignată', render: (item) => getParohieName(item.id_parohie) },
    { key: 'id_proterie', header: 'Protopopiat Asignat', render: (item) => getProtopopiatName(item.id_proterie) },
    { key: 'last_login', header: 'Ultima Conectare', render: (item) => item.last_login ? new Date(item.last_login).toLocaleString('ro-RO') : 'Niciodată'},
    {
      key: 'actions',
      header: 'Acțiuni',
      render: (item) => (
        <div className="space-x-2 flex">
          <Button variant="outline" size="sm" onClick={() => handleEdit(item)} aria-label={`Editează ${item.Nume}`}>
            <PencilIcon className="w-4 h-4 mr-1" />
            Edit
          </Button>
          <Button variant="danger" size="sm" onClick={() => handleDelete(item.id_user)} aria-label={`Șterge ${item.Nume}`}>
            <TrashIcon className="w-4 h-4 mr-1" />
            Șterge
          </Button>
        </div>
      ),
    },
  ];

  if (isLoading && users.length === 0) {
    return <LoadingSpinner message="Se încarcă utilizatorii..." size="lg" className="h-full"/>;
  }

  return (
    <div className="space-y-6">
      <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
        <h2 className="text-2xl sm:text-3xl font-bold text-slate-800">Administrare Utilizatori</h2>
        <Button variant="primary" onClick={handleAdd}>
           <PlusIcon className="w-5 h-5 mr-2" />
          Adaugă Utilizator
        </Button>
      </div>

      {alertInfo && <Alert type={alertInfo.type} message={alertInfo.message} onClose={() => setAlertInfo(null)} />}
      {error && !alertInfo && <Alert type="error" message={error} onClose={() => setError(null)} />} 
      
      {isSubmitting && <Alert type="info" message="Se procesează..." />}

      <Table columns={columns} data={users} getKey={(item) => item.id_user} caption="Lista Utilizatorilor Sistemului"/>

      <Modal
        isOpen={isModalOpen}
        onClose={() => { setIsModalOpen(false); setSelectedUser(null); }}
        title={selectedUser ? 'Editează Utilizator' : 'Adaugă Utilizator Nou'}
        size="xl" // Larger modal for user form
      >
        <UserForm
          initialData={selectedUser}
          onSubmit={handleFormSubmit}
          onCancel={() => { setIsModalOpen(false); setSelectedUser(null); }}
          roluri={roluri}
          parohii={parohii}
          protopopiate={protopopiate}
          isSubmitting={isSubmitting}
        />
      </Modal>
    </div>
  );
};

export default UsersPage;