
import { Protopopiat, User, Rol, Parohie, ProtopopiateStatus, UserStatus, DropdownOption } from '../types.tsx'; // Ensure .tsx
import { mockProtopopiate, mockUsers, mockRoluri, mockParohii } from './mockData.ts'; // Ensure .tsx

// Simulate API delay
const delay = (ms: number) => new Promise(resolve => setTimeout(resolve, ms));

// --- Protopopiate API ---
let currentProtopopiate = [...mockProtopopiate];
let nextProtopopiatId = currentProtopopiate.length > 0 ? Math.max(...currentProtopopiate.map(p => p.id_proterie)) + 1 : 1;

export const getProtopopiate = async (): Promise<Protopopiat[]> => {
  await delay(400 + Math.random() * 300);
  return JSON.parse(JSON.stringify(currentProtopopiate)); // Return copies
};

export const getProtopopiatById = async (id: number): Promise<Protopopiat | undefined> => {
  await delay(200 + Math.random() * 200);
  const found = currentProtopopiate.find(p => p.id_proterie === id);
  return found ? JSON.parse(JSON.stringify(found)) : undefined;
};

export const createProtopopiat = async (data: Omit<Protopopiat, 'id_proterie'>): Promise<Protopopiat> => {
  await delay(600 + Math.random() * 300);
  const newProtopopiat: Protopopiat = { ...data, id_proterie: nextProtopopiatId++ };
  currentProtopopiate.push(newProtopopiat);
  return JSON.parse(JSON.stringify(newProtopopiat));
};

export const updateProtopopiat = async (id: number, data: Partial<Protopopiat>): Promise<Protopopiat | undefined> => {
  await delay(600 + Math.random() * 300);
  const index = currentProtopopiate.findIndex(p => p.id_proterie === id);
  if (index !== -1) {
    currentProtopopiate[index] = { ...currentProtopopiate[index], ...data, id_proterie: id }; // Ensure ID isn't changed
    return JSON.parse(JSON.stringify(currentProtopopiate[index]));
  }
  return undefined;
};

export const deleteProtopopiat = async (id: number): Promise<boolean> => {
  await delay(400 + Math.random() * 200);
  const initialLength = currentProtopopiate.length;
  currentProtopopiate = currentProtopopiate.filter(p => p.id_proterie !== id);
  return currentProtopopiate.length < initialLength;
};

// --- Users API ---
let currentUsers = [...mockUsers];
let nextUserId = currentUsers.length > 0 ? Math.max(...currentUsers.map(u => u.id_user)) + 1 : 1;

export const getUsers = async (): Promise<User[]> => {
  await delay(400 + Math.random() * 300);
  return JSON.parse(JSON.stringify(currentUsers));
};

export const getUserById = async (id: number): Promise<User | undefined> => {
  await delay(200 + Math.random() * 200);
  const found = currentUsers.find(u => u.id_user === id);
  return found ? JSON.parse(JSON.stringify(found)) : undefined;
};

export const createUser = async (data: Omit<User, 'id_user' | 'last_login'>): Promise<User> => {
  await delay(600 + Math.random() * 300);
  const newUser: User = { 
    ...data, 
    id_user: nextUserId++, 
    last_login: new Date().toISOString() 
  };
  currentUsers.push(newUser);
  return JSON.parse(JSON.stringify(newUser));
};

export const updateUser = async (id: number, data: Partial<User>): Promise<User | undefined> => {
  await delay(600 + Math.random() * 300);
  const index = currentUsers.findIndex(u => u.id_user === id);
  if (index !== -1) {
    currentUsers[index] = { ...currentUsers[index], ...data, id_user: id }; // Ensure ID isn't changed
    // Password handling: if password field is present and empty, it means "don't change".
    // If it's present and non-empty, it's a new password.
    // If it's not present in `data`, it's also "don't change".
    // Mock API doesn't hash, but real API would.
    if (data.password === '') { // Explicitly empty string means don't update from form
        delete currentUsers[index].password; // Keep old one (not part of data)
    } else if(data.password) {
        currentUsers[index].password = data.password; // "Update" password
    }
    // Always remove password before sending back, unless it was part of the update `data`
    const userToReturn = { ...currentUsers[index] };
    if (!data.hasOwnProperty('password')) { // if password wasn't part of the partial update data
        delete userToReturn.password;
    }

    return JSON.parse(JSON.stringify(userToReturn));
  }
  return undefined;
};

export const deleteUser = async (id: number): Promise<boolean> => {
  await delay(400 + Math.random() * 200);
  const initialLength = currentUsers.length;
  currentUsers = currentUsers.filter(u => u.id_user !== id);
  return currentUsers.length < initialLength;
};

// --- Helper APIs for Dropdowns ---
export const getRoluri = async (): Promise<Rol[]> => {
  await delay(100 + Math.random() * 100);
  return JSON.parse(JSON.stringify(mockRoluri));
};

export const getParohiiList = async (): Promise<Parohie[]> => { // Renamed from getParohii to avoid conflict if a full Parohii CRUD is added
    await delay(100 + Math.random() * 100);
    return JSON.parse(JSON.stringify(mockParohii));
};

export const getProtopopiateList = async (): Promise<Protopopiat[]> => { // Renamed for consistency
    await delay(100 + Math.random() * 100);
    return JSON.parse(JSON.stringify(currentProtopopiate)); // Use the dynamic list
};

// Simpler way to get DropdownOption[] from string enums
export const getProtopopiateStatusOptions = (): DropdownOption[] => {
    return Object.values(ProtopopiateStatus).map(statusValue => ({ 
      value: statusValue, 
      label: statusValue.charAt(0).toUpperCase() + statusValue.slice(1) // Capitalize for display e.g. "Activ"
    }));
}

export const getUserStatusOptions = (): DropdownOption[] => {
    return Object.values(UserStatus).map(statusValue => ({ 
      value: statusValue, 
      label: statusValue.charAt(0).toUpperCase() + statusValue.slice(1) // Capitalize for display
    }));
}
